#!/bin/bash

################################################################################
# Oasis Investor Database System - Create Download Package
# Version 2.1
# This script creates a clean downloadable package
################################################################################

echo "======================================================================"
echo "  Creating Oasis Investor Database System v2.1 Download Package"
echo "======================================================================"
echo ""

# Set variables
VERSION="2.1"
PACKAGE_NAME="oasis-investor-database-v${VERSION}"
BUILD_DIR="build"
PACKAGE_DIR="${BUILD_DIR}/${PACKAGE_NAME}"
OUTPUT_FILE="${PACKAGE_NAME}.zip"

# Create build directory
echo "[1/6] Creating build directory..."
rm -rf ${BUILD_DIR}
mkdir -p ${PACKAGE_DIR}

# Copy core files
echo "[2/6] Copying core application files..."
mkdir -p ${PACKAGE_DIR}/css
mkdir -p ${PACKAGE_DIR}/database
mkdir -p ${PACKAGE_DIR}/forms
mkdir -p ${PACKAGE_DIR}/includes
mkdir -p ${PACKAGE_DIR}/image/logo

# Copy PHP files
cp index.php ${PACKAGE_DIR}/
cp test_connection.php ${PACKAGE_DIR}/

# Copy forms
cp forms/client_form.php ${PACKAGE_DIR}/forms/
cp forms/search_clients.php ${PACKAGE_DIR}/forms/
cp forms/contact_records.php ${PACKAGE_DIR}/forms/
cp forms/staff_management.php ${PACKAGE_DIR}/forms/
cp forms/introducer_form.php ${PACKAGE_DIR}/forms/
cp forms/introducers_list.php ${PACKAGE_DIR}/forms/

# Copy includes
cp includes/config.php ${PACKAGE_DIR}/includes/
cp includes/db.php ${PACKAGE_DIR}/includes/
cp includes/header.php ${PACKAGE_DIR}/includes/
cp includes/footer.php ${PACKAGE_DIR}/includes/

# Copy CSS
cp css/style.css ${PACKAGE_DIR}/css/

# Copy database
cp database/schema.sql ${PACKAGE_DIR}/database/
cp database/sample_data.sql ${PACKAGE_DIR}/database/
cp database/add_introducers.sql ${PACKAGE_DIR}/database/

# Copy documentation
echo "[3/6] Copying documentation..."
cp README.md ${PACKAGE_DIR}/
cp START_HERE.md ${PACKAGE_DIR}/
cp SETUP_INSTRUCTIONS.txt ${PACKAGE_DIR}/
cp UPLOAD_CHECKLIST.txt ${PACKAGE_DIR}/
cp QUICK_REFERENCE.md ${PACKAGE_DIR}/
cp SYSTEM_OVERVIEW.md ${PACKAGE_DIR}/
cp DESIGN_IMPROVEMENTS.md ${PACKAGE_DIR}/
cp BRANDING_SETUP.md ${PACKAGE_DIR}/
cp BRANDING_CHANGES_SUMMARY.md ${PACKAGE_DIR}/
cp FILE_MANIFEST.txt ${PACKAGE_DIR}/
cp VERSION.txt ${PACKAGE_DIR}/
cp DOWNLOAD_PACKAGE.md ${PACKAGE_DIR}/
cp CHANGES_v2.1_SUMMARY.md ${PACKAGE_DIR}/
cp INSTALL_v2.1_UPDATE.txt ${PACKAGE_DIR}/
cp database/UPDATE_v2.1_INTRODUCERS.md ${PACKAGE_DIR}/database/

# Copy placeholder files
echo "[4/6] Copying placeholder files..."
cp PLACE_FAVICON_HERE.txt ${PACKAGE_DIR}/
cp image/logo/PLACE_LOGO_HERE.txt ${PACKAGE_DIR}/image/logo/
cp image/README.md ${PACKAGE_DIR}/image/

# Create .htaccess for Apache (optional security)
echo "[5/6] Creating security files..."
cat > ${PACKAGE_DIR}/.htaccess << 'EOF'
# Oasis Investor Database System - Apache Configuration
# Disable directory browsing
Options -Indexes

# Protect sensitive files
<FilesMatch "^(config\.php|\.htaccess)$">
    Order allow,deny
    Deny from all
</FilesMatch>

# Set default charset
AddDefaultCharset UTF-8

# Enable compression (if mod_deflate is available)
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>
EOF

# Create robots.txt
cat > ${PACKAGE_DIR}/robots.txt << 'EOF'
# Oasis Investor Database System
# This is a private application - disallow all web crawlers
User-agent: *
Disallow: /
EOF

# Create README for package
cat > ${PACKAGE_DIR}/READ_ME_FIRST.txt << 'EOF'
================================================================================
OASIS INVESTOR DATABASE SYSTEM v2.0 - DOWNLOAD PACKAGE
================================================================================

Thank you for downloading the Oasis Investor Database System!

QUICK START:
-----------
1. Read START_HERE.md for complete getting started guide
2. Read UPLOAD_CHECKLIST.txt for step-by-step deployment
3. Follow the setup instructions carefully

IMPORTANT - BEFORE YOU BEGIN:
-----------------------------
You must prepare TWO files for upload:

1. OCFLoogo.png - Your company logo (200x50px PNG)
   Upload to: /image/logo/OCFLoogo.png

2. favicon.png - Your site favicon (32x32px PNG)
   Upload to: /favicon.png (root directory)

See BRANDING_SETUP.md for detailed specifications.

INSTALLATION OVERVIEW:
---------------------
1. Upload all files to your web server
2. Upload logo and favicon files
3. Import database/schema.sql to MySQL/MariaDB
4. Configure includes/config.php with database credentials
5. Visit test_connection.php to verify setup
6. Access index.php to start using the system

DOCUMENTATION:
-------------
⭐ START_HERE.md - Begin here!
📋 UPLOAD_CHECKLIST.txt - Step-by-step guide
📖 README.md - Complete documentation
🎨 BRANDING_SETUP.md - Logo & favicon setup
⚡ QUICK_REFERENCE.md - Daily operations
🔧 SYSTEM_OVERVIEW.md - Technical details

SYSTEM REQUIREMENTS:
-------------------
- PHP 7.4 or higher
- MySQL 5.7+ or MariaDB 10.x+
- Apache or Nginx web server
- Modern web browser

PACKAGE CONTENTS:
----------------
✓ 10 PHP application files
✓ 2 SQL database files
✓ 1 CSS stylesheet
✓ 12 documentation files
✓ Directory structure
✓ Security files

SUPPORT:
-------
All support information is in the documentation files.
Start with START_HERE.md for quickest results.

VERSION: 2.0 - Oasis Edition
RELEASE: December 2024
STATUS: Production Ready ✓

Good luck with your deployment!

================================================================================
EOF

# Create package
echo "[6/6] Creating ZIP package..."
cd ${BUILD_DIR}
zip -r ../${OUTPUT_FILE} ${PACKAGE_NAME}/ -q
cd ..

# Cleanup
echo ""
echo "Cleaning up build directory..."
# rm -rf ${BUILD_DIR}

# Calculate size
FILESIZE=$(du -h ${OUTPUT_FILE} | cut -f1)

# Done
echo ""
echo "======================================================================"
echo "  ✓ Package created successfully!"
echo "======================================================================"
echo ""
echo "  File: ${OUTPUT_FILE}"
echo "  Size: ${FILESIZE}"
echo "  Location: $(pwd)/${OUTPUT_FILE}"
echo ""
echo "  Package contents:"
echo "    - 12 PHP files (application)"
echo "    - 3 SQL files (database)"
echo "    - 1 CSS file (styling)"
echo "    - 15 documentation files"
echo "    - Directory structure"
echo "    - Security files (.htaccess, robots.txt)"
echo ""
echo "  Ready for distribution!"
echo ""
echo "======================================================================"
echo ""

# Display next steps
echo "Next steps:"
echo "1. Test the package by extracting and deploying"
echo "2. Verify all files are included"
echo "3. Share ${OUTPUT_FILE} with users"
echo ""
echo "Users should start with: START_HERE.md"
echo ""
