# Investor Database System - Technical Overview

## System Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                    PRESENTATION LAYER                        │
│  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌──────────┐   │
│  │Dashboard │  │  Client  │  │  Search  │  │ Contact  │   │
│  │index.php │  │   Form   │  │  Clients │  │ Records  │   │
│  └──────────┘  └──────────┘  └──────────┘  └──────────┘   │
│                                                               │
│  ┌──────────┐  ┌─────────────────────────────────────┐     │
│  │  Staff   │  │      Common Components               │     │
│  │Mgmt Form │  │  - header.php  - footer.php          │     │
│  └──────────┘  │  - Navigation  - Styling (CSS)       │     │
│                 └─────────────────────────────────────┘     │
└─────────────────────────────────────────────────────────────┘
                            ↓
┌─────────────────────────────────────────────────────────────┐
│                    APPLICATION LAYER                         │
│  ┌──────────────────────────────────────────────────┐       │
│  │  includes/db.php - Database Connection           │       │
│  │  - Connection management                          │       │
│  │  - Input sanitization functions                   │       │
│  │  - Utility functions (age calculation, etc.)      │       │
│  └──────────────────────────────────────────────────┘       │
│  ┌──────────────────────────────────────────────────┐       │
│  │  includes/config.php - Configuration             │       │
│  │  - Database credentials                           │       │
│  │  - Application settings                           │       │
│  └──────────────────────────────────────────────────┘       │
└─────────────────────────────────────────────────────────────┘
                            ↓
┌─────────────────────────────────────────────────────────────┐
│                      DATA LAYER                              │
│  ┌────────────┐  ┌────────────┐  ┌────────────────┐        │
│  │   staff    │  │  clients   │  │ contact_records│        │
│  │   table    │  │   table    │  │     table      │        │
│  └────────────┘  └────────────┘  └────────────────┘        │
│         ↓              ↓                  ↓                  │
│  MariaDB / MySQL Database (investor_db)                     │
└─────────────────────────────────────────────────────────────┘
```

## Database Schema

### 1. Staff Table
**Purpose:** Stores bank employee information

| Field | Type | Description |
|-------|------|-------------|
| staff_id | INT (PK) | Unique staff identifier |
| first_name | VARCHAR(100) | Staff first name |
| last_name | VARCHAR(100) | Staff last name |
| email | VARCHAR(150) | Unique email address |
| phone | VARCHAR(20) | Contact phone |
| department | VARCHAR(100) | Department name |
| position | VARCHAR(100) | Job position |
| active | TINYINT(1) | Active status (1/0) |
| created_at | TIMESTAMP | Record creation date |
| updated_at | TIMESTAMP | Last update date |

**Relationships:**
- One-to-Many with `clients` (assigned_staff_id)
- One-to-Many with `contact_records` (staff_id)

### 2. Clients Table
**Purpose:** Comprehensive investor/client records

| Section | Fields | Description |
|---------|--------|-------------|
| **Contact Info** | first_name, last_name, email, phone, mobile, address fields, country | Basic contact details |
| **KYC** | kyc_status, kyc_dates, id_type, id_number, tax_id, pep_status, risk_rating | Due diligence data |
| **Personal** | date_of_birth, age, nationality, occupation, investor_profile | Personal information |
| **Financial** | annual_income, source_of_income, total_net_worth, liquid_assets | Financial status |
| **Investment** | current_investments, investment_experience, investment_interests, risk_tolerance, investment_horizon | Investment profile |
| **Meta** | client_status, assigned_staff_id, notes, timestamps | Administrative data |

**Key Features:**
- Automatic age calculation from date_of_birth
- Currency fields for multi-currency support
- ENUM fields for controlled values
- Full-text search capabilities on investment_interests

**Relationships:**
- Many-to-One with `staff` (assigned_staff_id)
- One-to-Many with `contact_records` (client_id)

### 3. Contact Records Table
**Purpose:** Track all client interactions

| Field | Type | Description |
|-------|------|-------------|
| contact_id | INT (PK) | Unique contact identifier |
| client_id | INT (FK) | Related client |
| staff_id | INT (FK) | Staff member who made contact |
| contact_date | TIMESTAMP | When contact occurred |
| contact_type | ENUM | Newsletter, Phone Call, Visit, Email, Video Conference, Other |
| subject | VARCHAR(200) | Contact subject |
| description | TEXT | Detailed description |
| outcome | TEXT | Result of contact |
| follow_up_required | TINYINT(1) | Follow-up needed flag |
| follow_up_date | DATE | Scheduled follow-up date |
| created_at | TIMESTAMP | Record creation |

**Relationships:**
- Many-to-One with `clients` (client_id) - CASCADE DELETE
- Many-to-One with `staff` (staff_id) - RESTRICT DELETE

## Core Features by Module

### Dashboard (index.php)
**Statistics Display:**
- Total clients count
- Active clients count
- Prospect clients count
- Total contact records
- Pending KYC count

**Recent Activity:**
- Last 10 clients added
- Last 10 contact records
- Quick access links

**Technologies:**
- PHP queries with COUNT aggregation
- Color-coded stat cards with gradients
- Responsive grid layout

### Client Form (forms/client_form.php)
**Functionality:**
- Add new clients
- Edit existing clients
- Automatic age calculation
- Staff assignment dropdown

**Form Sections:**
1. Contact Information (11 fields)
2. KYC Due Diligence (9 fields)
3. Personal Information (4 fields + profile)
4. Financial Information (6 fields)
5. Investment Information (6 fields)
6. Additional Information (3 fields + notes)

**Validation:**
- Required fields: first_name, last_name, email
- Email format validation
- Number validation for financial fields
- Date validation

**Database Operations:**
- INSERT for new clients
- UPDATE for existing clients
- Auto-calculation of age from DOB
- Foreign key relationship to staff table

### Search Form (forms/search_clients.php)
**Search Criteria:**
1. Text search (name/email)
2. Client status dropdown
3. KYC status dropdown
4. Risk rating dropdown
5. Investment experience dropdown
6. Risk tolerance dropdown
7. Investment horizon dropdown
8. Country dropdown (populated from existing data)
9. Assigned staff dropdown
10. Investment interests keyword search
11. Net worth range (min/max)
12. Age range (min/max)

**Search Logic:**
- Dynamic WHERE clause building
- Multiple criteria with AND logic
- LIKE queries for text fields
- Range queries for numeric fields
- JOIN with staff table for staff names

**Results Display:**
- Paginated table view
- Show: ID, Name, Email, Status, KYC, Age, Net Worth, Interests, Staff
- Quick "View" button to edit form
- Result count display

### Contact Records (forms/contact_records.php)
**Add Contact:**
- Client selection dropdown
- Staff member dropdown (auto-populated from active staff)
- Contact type dropdown (Newsletter, Phone Call, Visit, Email, Video Conference, Other)
- DateTime picker for contact date
- Subject and description fields
- Outcome tracking
- Follow-up flag and date

**View Contacts:**
- All contacts or filter by specific client
- Sortable table view
- Modal popup for full contact details
- Follow-up indicators with date

**Database Operations:**
- INSERT new contact records
- UPDATE client's last_contact_date
- JOIN queries to display client and staff names

### Staff Management (forms/staff_management.php)
**Features:**
- Add new staff members
- Edit existing staff
- Active/inactive status toggle
- View assigned client count per staff

**Fields:**
- Name, email, phone
- Department, position
- Active status

**Display:**
- Table showing all staff with statistics
- Edit mode with pre-filled form
- Client count per staff member (JOIN query)

**Business Logic:**
- Active staff appear in dropdowns
- Inactive staff retained for historical records
- Cannot delete staff with assigned clients (foreign key constraint)

## Security Features

### Current Implementation
1. **Input Sanitization:**
   - `sanitize_input()` function in db.php
   - Trim, stripslashes, htmlspecialchars
   - MySQL real_escape_string

2. **SQL Injection Prevention:**
   - All user inputs sanitized
   - Special characters escaped

3. **XSS Prevention:**
   - htmlspecialchars on all output
   - Proper encoding of user-generated content

4. **Database Security:**
   - Foreign key constraints
   - CASCADE/RESTRICT delete rules
   - UNIQUE constraints on emails

### Recommended Enhancements for Production
1. **Authentication System:**
   - User login/logout
   - Session management
   - Password hashing (bcrypt/argon2)

2. **Authorization:**
   - Role-based access control (admin, advisor, viewer)
   - Permission checks per action

3. **Prepared Statements:**
   - Replace string concatenation with PDO/mysqli prepared statements
   - Parameterized queries

4. **HTTPS:**
   - SSL/TLS certificate
   - Force HTTPS redirect

5. **Audit Trail:**
   - Log all modifications
   - Track who changed what and when

6. **Data Encryption:**
   - Encrypt sensitive fields (tax_id, id_number)
   - At-rest encryption for database

## Performance Considerations

### Current Optimizations
1. **Database Indexes:**
   - Primary keys on all tables
   - Indexes on foreign keys
   - Index on client status and KYC status
   - Index on investment_interests (first 100 chars)

2. **Efficient Queries:**
   - JOIN optimization
   - SELECT only needed columns
   - Limit results where appropriate

### Scalability Recommendations
1. **Query Optimization:**
   - Add EXPLAIN to analyze queries
   - Implement query caching
   - Paginate large result sets

2. **Database:**
   - Regular OPTIMIZE TABLE
   - Monitor slow query log
   - Consider read replicas for high traffic

3. **Application:**
   - Implement Redis/Memcached for session storage
   - Cache frequently accessed data
   - Minify CSS/JS

4. **Infrastructure:**
   - Load balancer for multiple web servers
   - CDN for static assets
   - Database connection pooling

## File Structure Details

```
investor-database/
├── .same/                      # Development tracking
│   └── todos.md                # Task list
├── css/
│   └── style.css              # Professional styling (responsive)
├── database/
│   ├── schema.sql             # Database structure + sample staff
│   └── sample_data.sql        # Optional demo clients & contacts
├── forms/
│   ├── client_form.php        # Add/Edit clients (comprehensive)
│   ├── search_clients.php     # Advanced search with filters
│   ├── contact_records.php    # Contact tracking & history
│   └── staff_management.php   # Staff CRUD operations
├── includes/
│   ├── config.php             # Database & app configuration
│   ├── db.php                 # Connection + helper functions
│   ├── header.php             # Common header + navigation
│   └── footer.php             # Common footer
├── index.php                  # Dashboard (main page)
├── test_connection.php        # Database setup verification
├── README.md                  # User documentation
├── SETUP_INSTRUCTIONS.txt     # Quick setup guide
└── SYSTEM_OVERVIEW.md         # This file (technical docs)
```

## Technology Stack

- **Backend:** PHP 7.4+ (tested with PHP 8.x)
- **Database:** MariaDB 10.x / MySQL 5.7+
- **Frontend:** HTML5, CSS3, JavaScript (Vanilla)
- **Web Server:** Apache / Nginx / PHP Built-in Server
- **Character Set:** UTF-8 (utf8mb4)
- **Storage Engine:** InnoDB (for foreign key support)

## Browser Requirements

- Modern browsers with JavaScript enabled
- Responsive design supports mobile, tablet, desktop
- Tested on Chrome 90+, Firefox 88+, Safari 14+, Edge 90+

## Data Flow Examples

### Adding a New Client
1. User fills form at `forms/client_form.php`
2. Form submits via POST
3. PHP sanitizes all inputs (`sanitize_input()`)
4. Age auto-calculated from date_of_birth
5. INSERT query executed
6. Success message displayed
7. Redirects to edit mode with new client_id
8. User can immediately add contact records

### Searching for Clients
1. User selects criteria in `forms/search_clients.php`
2. Form submits via POST
3. Dynamic WHERE clause built from non-empty fields
4. JOIN query with staff table
5. Results displayed in table
6. Click "View" opens client in edit form
7. Can filter results further or clear search

### Recording a Contact
1. Select client from dropdown (or pre-selected if coming from client form)
2. Select staff member (active staff only)
3. Choose contact type from predefined options
4. Enter date, subject, description, outcome
5. Optional: set follow-up flag and date
6. Submit via POST
7. INSERT into contact_records
8. UPDATE clients.last_contact_date
9. Confirmation message shown
10. Contact appears in history table below

## Customization Guide

### Adding New Client Fields
1. Alter `clients` table: `ALTER TABLE clients ADD COLUMN new_field VARCHAR(100);`
2. Add field to `client_form.php` form
3. Add to INSERT and UPDATE queries
4. Optionally add to search form
5. Update display in search results/dashboard

### Adding New Contact Types
1. Alter enum in `contact_records` table:
   ```sql
   ALTER TABLE contact_records
   MODIFY contact_type ENUM('Newsletter','Phone Call','Visit','Email','Video Conference','Other','New Type');
   ```
2. Add option to dropdown in `contact_records.php`

### Changing Color Scheme
Edit `css/style.css`:
- Primary color: `#3498db` (blue)
- Secondary color: `#2c3e50` (dark blue)
- Success: `#27ae60` (green)
- Danger: `#e74c3c` (red)

### Adding User Authentication
1. Create `users` table with hashed passwords
2. Create `login.php` with form
3. Add `session_start()` to header.php
4. Check `$_SESSION['user_id']` on all pages
5. Redirect to login if not authenticated
6. Add logout functionality

## Maintenance Tasks

### Daily
- Monitor error logs
- Check backup completion

### Weekly
- Review contact follow-ups
- Check expired KYC records

### Monthly
- Database optimization: `OPTIMIZE TABLE clients, contact_records, staff;`
- Review and archive old records
- Update staff list

### Quarterly
- Security audit
- Performance review
- User feedback collection

## Troubleshooting Guide

**Problem:** Can't connect to database
- Check MySQL/MariaDB is running
- Verify credentials in `config.php`
- Test with `test_connection.php`

**Problem:** Clients not appearing
- Check if records exist: `SELECT COUNT(*) FROM clients;`
- Verify query in index.php has no errors
- Check error logs

**Problem:** Dropdown not showing staff
- Verify staff.active = 1
- Check if staff table has records
- Inspect SQL query in form

**Problem:** Search returns no results
- Check if WHERE conditions too restrictive
- Test with single criterion at a time
- Verify data exists matching criteria

## Support and Further Development

### Planned Features (Future Versions)
- [ ] Document upload/attachment system
- [ ] Email integration (send newsletters)
- [ ] Report generation (PDF/Excel)
- [ ] Calendar integration for follow-ups
- [ ] Multi-language support
- [ ] API for third-party integrations
- [ ] Advanced analytics dashboard
- [ ] Client portal (self-service)

### Contributing
This is a proprietary system. Contact the development team for customization requests.

---

**Document Version:** 1.0
**Last Updated:** December 2024
**System Version:** 1.0
**Author:** Investment Bank Development Team
