-- OPTION A: Merge with Existing Tables
-- This script updates your existing users, user_sessions, and activity_log tables
-- to work with the new authentication system
-- BACKUP FIRST!

USE oasiscapfin_investor_db;

-- ============================================================================
-- BACKUP REMINDER
-- ============================================================================
SELECT '⚠️  CRITICAL: Have you backed up your database? (Y/N)' as 'BACKUP CHECK';
SELECT 'If NO, stop now and run: mysqldump -u username -p oasiscapfin_investor_db > backup.sql' as 'INSTRUCTION';
SELECT 'Press any key to continue if backup is complete...' as '';

-- ============================================================================
-- UPDATE EXISTING USERS TABLE
-- ============================================================================

SELECT '========================================' as '';
SELECT 'UPDATING USERS TABLE' as 'STEP 1';
SELECT '========================================' as '';

-- Add missing columns to existing users table if they don't exist
SET @dbname = 'oasiscapfin_investor_db';
SET @tablename = 'users';

-- Add role column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'role');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN role ENUM(''Admin'', ''Manager'', ''User'', ''Viewer'') DEFAULT ''User'' AFTER email',
    'SELECT ''role column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add staff_id column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'staff_id');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN staff_id INT NULL AFTER role',
    'SELECT ''staff_id column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add status column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'status');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN status ENUM(''Active'', ''Inactive'', ''Suspended'', ''Locked'') DEFAULT ''Active'' AFTER staff_id',
    'SELECT ''status column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add failed_login_attempts column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'failed_login_attempts');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN failed_login_attempts INT DEFAULT 0 AFTER status',
    'SELECT ''failed_login_attempts column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add last_failed_login column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'last_failed_login');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN last_failed_login TIMESTAMP NULL AFTER failed_login_attempts',
    'SELECT ''last_failed_login column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add last_login column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'last_login');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN last_login TIMESTAMP NULL AFTER last_failed_login',
    'SELECT ''last_login column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add last_password_change column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'last_password_change');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN last_password_change TIMESTAMP DEFAULT CURRENT_TIMESTAMP AFTER last_login',
    'SELECT ''last_password_change column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add remember_token column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'remember_token');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE users ADD COLUMN remember_token VARCHAR(64) NULL AFTER last_password_change',
    'SELECT ''remember_token column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes if missing
SET @index_exists = (SELECT COUNT(*) FROM information_schema.statistics
    WHERE table_schema = @dbname AND table_name = @tablename AND index_name = 'idx_role');
SET @query = IF(@index_exists = 0,
    'ALTER TABLE users ADD INDEX idx_role (role)',
    'SELECT ''idx_role index already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @index_exists = (SELECT COUNT(*) FROM information_schema.statistics
    WHERE table_schema = @dbname AND table_name = @tablename AND index_name = 'idx_status');
SET @query = IF(@index_exists = 0,
    'ALTER TABLE users ADD INDEX idx_status (status)',
    'SELECT ''idx_status index already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign key to staff if missing and staff table exists
SET @fk_exists = (SELECT COUNT(*) FROM information_schema.key_column_usage
    WHERE table_schema = @dbname AND table_name = @tablename
    AND constraint_name LIKE '%staff%' AND referenced_table_name = 'staff');
SET @staff_table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = 'staff');
SET @query = IF(@fk_exists = 0 AND @staff_table_exists > 0,
    'ALTER TABLE users ADD FOREIGN KEY (staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL',
    'SELECT ''Foreign key to staff already exists or staff table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Create default admin if no users exist or no admin exists
INSERT INTO users (username, email, password_hash, first_name, last_name, role, status, email_verified)
SELECT 'admin', 'admin@oasis.local', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
       'System', 'Administrator', 'Admin', 'Active', 1
WHERE NOT EXISTS (SELECT 1 FROM users WHERE username = 'admin' OR role = 'Admin');

SELECT 'Users table updated successfully!' as Status;

-- ============================================================================
-- UPDATE EXISTING USER_SESSIONS TABLE
-- ============================================================================

SELECT '========================================' as '';
SELECT 'UPDATING USER_SESSIONS TABLE' as 'STEP 2';
SELECT '========================================' as '';

SET @tablename = 'user_sessions';

-- Add expires_at column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'expires_at');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE user_sessions ADD COLUMN expires_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP',
    'SELECT ''expires_at column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes if missing
SET @index_exists = (SELECT COUNT(*) FROM information_schema.statistics
    WHERE table_schema = @dbname AND table_name = @tablename AND index_name = 'idx_expires');
SET @query = IF(@index_exists = 0,
    'ALTER TABLE user_sessions ADD INDEX idx_expires (expires_at)',
    'SELECT ''idx_expires index already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SELECT 'User_sessions table updated successfully!' as Status;

-- ============================================================================
-- UPDATE EXISTING ACTIVITY_LOG TABLE
-- ============================================================================

SELECT '========================================' as '';
SELECT 'UPDATING ACTIVITY_LOG TABLE' as 'STEP 3';
SELECT '========================================' as '';

SET @tablename = 'activity_log';

-- Add entity_type column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'entity_type');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE activity_log ADD COLUMN entity_type VARCHAR(50) NULL AFTER action',
    'SELECT ''entity_type column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add entity_id column if missing
SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'entity_id');
SET @query = IF(@col_exists = 0,
    'ALTER TABLE activity_log ADD COLUMN entity_id INT NULL AFTER entity_type',
    'SELECT ''entity_id column already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes if missing
SET @index_exists = (SELECT COUNT(*) FROM information_schema.statistics
    WHERE table_schema = @dbname AND table_name = @tablename AND index_name = 'idx_entity');
SET @query = IF(@index_exists = 0,
    'ALTER TABLE activity_log ADD INDEX idx_entity (entity_type, entity_id)',
    'SELECT ''idx_entity index already exists'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SELECT 'Activity_log table updated successfully!' as Status;

-- ============================================================================
-- CREATE LOGIN_ATTEMPTS TABLE (This one probably doesn't exist)
-- ============================================================================

SELECT '========================================' as '';
SELECT 'CREATING LOGIN_ATTEMPTS TABLE' as 'STEP 4';
SELECT '========================================' as '';

CREATE TABLE IF NOT EXISTS login_attempts (
    attempt_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45),
    success TINYINT(1) NOT NULL,
    failure_reason VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_username (username),
    INDEX idx_ip (ip_address),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SELECT 'Login_attempts table created successfully!' as Status;

-- ============================================================================
-- ADD TRACKING COLUMNS TO EXISTING TABLES
-- ============================================================================

SELECT '========================================' as '';
SELECT 'ADDING TRACKING COLUMNS' as 'STEP 5';
SELECT '========================================' as '';

-- Add to clients table if it exists
SET @tablename = 'clients';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE clients ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at',
    'SELECT ''Tracking columns already exist in clients or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to contact_records table if it exists
SET @tablename = 'contact_records';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE contact_records ADD COLUMN created_by INT NULL AFTER created_at',
    'SELECT ''Tracking column already exists in contact_records or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to staff table if it exists
SET @tablename = 'staff';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE staff ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at',
    'SELECT ''Tracking columns already exist in staff or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to introducers table if it exists
SET @tablename = 'introducers';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE introducers ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at',
    'SELECT ''Tracking columns already exist in introducers or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SELECT 'Tracking columns added successfully!' as Status;

-- ============================================================================
-- CREATE USER ACTIVITY VIEW
-- ============================================================================

SELECT '========================================' as '';
SELECT 'CREATING ACTIVITY VIEW' as 'STEP 6';
SELECT '========================================' as '';

CREATE OR REPLACE VIEW user_activity_summary AS
SELECT
    u.user_id,
    u.username,
    u.first_name,
    u.last_name,
    u.role,
    u.last_login,
    COUNT(DISTINCT al.log_id) as total_actions,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN al.log_id END) as actions_last_7_days,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN al.log_id END) as actions_last_30_days
FROM users u
LEFT JOIN activity_log al ON u.user_id = al.user_id
GROUP BY u.user_id, u.username, u.first_name, u.last_name, u.role, u.last_login;

SELECT 'View created successfully!' as Status;

-- ============================================================================
-- FINAL STATUS
-- ============================================================================

SELECT '========================================' as '';
SELECT '✓ MIGRATION COMPLETE!' as 'RESULT';
SELECT '========================================' as '';

SELECT 'Existing tables have been updated to work with authentication system' as Status;
SELECT 'Next steps:' as '';
SELECT '1. Upload PHP files (auth.php, login.php, etc.)' as Step;
SELECT '2. Modify includes/header.php' as Step;
SELECT '3. Test login with admin/Admin@123' as Step;
SELECT '4. Change default password immediately!' as Step;

SELECT '========================================' as '';
SELECT 'Default Admin Credentials:' as '';
SELECT 'Username: admin' as '';
SELECT 'Password: Admin@123' as '';
SELECT 'CHANGE IMMEDIATELY AFTER LOGIN!' as '';
SELECT '========================================' as '';
