-- OPTION B: Independent Tables (Renamed to Avoid Conflicts)
-- This creates new tables with "auth_" prefix to run independently
-- Your existing file upload system remains untouched

USE oasiscapfin_investor_db;

-- ============================================================================
-- CREATE INDEPENDENT AUTHENTICATION TABLES
-- ============================================================================

SELECT '========================================' as '';
SELECT 'CREATING INDEPENDENT AUTH TABLES' as 'STEP 1';
SELECT '========================================' as '';

-- Create auth_users table (independent from existing users table)
CREATE TABLE IF NOT EXISTS auth_users (
    user_id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,

    -- Personal Information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),

    -- User Role & Permissions
    role ENUM('Admin', 'Manager', 'User', 'Viewer') DEFAULT 'User',
    staff_id INT NULL,

    -- Account Status
    status ENUM('Active', 'Inactive', 'Suspended', 'Locked') DEFAULT 'Active',
    email_verified TINYINT(1) DEFAULT 0,

    -- Security
    failed_login_attempts INT DEFAULT 0,
    last_failed_login TIMESTAMP NULL,
    last_login TIMESTAMP NULL,
    last_password_change TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    password_reset_token VARCHAR(64) NULL,
    password_reset_expires TIMESTAMP NULL,

    -- Session Management
    remember_token VARCHAR(64) NULL,

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_by INT NULL,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    updated_by INT NULL,

    FOREIGN KEY (staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SELECT 'auth_users table created!' as Status;

-- Create auth_sessions table (independent from existing user_sessions)
CREATE TABLE IF NOT EXISTS auth_sessions (
    session_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    session_token VARCHAR(64) UNIQUE NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES auth_users(user_id) ON DELETE CASCADE,
    INDEX idx_session_token (session_token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SELECT 'auth_sessions table created!' as Status;

-- Create auth_activity_log table (independent from existing activity_log)
CREATE TABLE IF NOT EXISTS auth_activity_log (
    log_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) NULL,
    entity_id INT NULL,
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES auth_users(user_id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_action (action),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SELECT 'auth_activity_log table created!' as Status;

-- Create auth_login_attempts table
CREATE TABLE IF NOT EXISTS auth_login_attempts (
    attempt_id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45),
    success TINYINT(1) NOT NULL,
    failure_reason VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_username (username),
    INDEX idx_ip (ip_address),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SELECT 'auth_login_attempts table created!' as Status;

-- ============================================================================
-- INSERT DEFAULT USERS
-- ============================================================================

SELECT '========================================' as '';
SELECT 'CREATING DEFAULT USERS' as 'STEP 2';
SELECT '========================================' as '';

-- Insert default admin user
-- Password: Admin@123 (MUST BE CHANGED IMMEDIATELY)
INSERT INTO auth_users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified
) VALUES (
    'admin',
    'admin@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'System',
    'Administrator',
    'Admin',
    'Active',
    1
);

-- Insert sample manager user
-- Password: Manager@123 (MUST BE CHANGED)
INSERT INTO auth_users (
    username, email, password_hash, first_name, last_name,
    role, status, email_verified, staff_id
) VALUES (
    'manager',
    'manager@oasis.local',
    '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi',
    'John',
    'Manager',
    'Manager',
    'Active',
    1,
    1
);

SELECT '2 default users created!' as Status;

-- ============================================================================
-- ADD TRACKING COLUMNS TO EXISTING TABLES
-- ============================================================================

SELECT '========================================' as '';
SELECT 'ADDING TRACKING COLUMNS' as 'STEP 3';
SELECT '========================================' as '';

-- Add to clients table if it exists and doesn't have tracking
SET @dbname = 'oasiscapfin_investor_db';
SET @tablename = 'clients';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE clients ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at, ADD FOREIGN KEY (created_by) REFERENCES auth_users(user_id) ON DELETE SET NULL, ADD FOREIGN KEY (updated_by) REFERENCES auth_users(user_id) ON DELETE SET NULL',
    'SELECT ''Tracking columns already exist in clients or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to contact_records table
SET @tablename = 'contact_records';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE contact_records ADD COLUMN created_by INT NULL AFTER created_at, ADD FOREIGN KEY (created_by) REFERENCES auth_users(user_id) ON DELETE SET NULL',
    'SELECT ''Tracking column already exists in contact_records or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to staff table
SET @tablename = 'staff';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE staff ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at, ADD FOREIGN KEY (created_by) REFERENCES auth_users(user_id) ON DELETE SET NULL, ADD FOREIGN KEY (updated_by) REFERENCES auth_users(user_id) ON DELETE SET NULL',
    'SELECT ''Tracking columns already exist in staff or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add to introducers table
SET @tablename = 'introducers';
SET @table_exists = (SELECT COUNT(*) FROM information_schema.tables
    WHERE table_schema = @dbname AND table_name = @tablename);

SET @col_exists = (SELECT COUNT(*) FROM information_schema.columns
    WHERE table_schema = @dbname AND table_name = @tablename AND column_name = 'created_by');
SET @query = IF(@table_exists > 0 AND @col_exists = 0,
    'ALTER TABLE introducers ADD COLUMN created_by INT NULL AFTER created_at, ADD COLUMN updated_by INT NULL AFTER updated_at, ADD FOREIGN KEY (created_by) REFERENCES auth_users(user_id) ON DELETE SET NULL, ADD FOREIGN KEY (updated_by) REFERENCES auth_users(user_id) ON DELETE SET NULL',
    'SELECT ''Tracking columns already exist in introducers or table not found'' as Status');
PREPARE stmt FROM @query;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SELECT 'Tracking columns added!' as Status;

-- ============================================================================
-- CREATE VIEW
-- ============================================================================

SELECT '========================================' as '';
SELECT 'CREATING VIEW' as 'STEP 4';
SELECT '========================================' as '';

CREATE OR REPLACE VIEW auth_user_activity_summary AS
SELECT
    u.user_id,
    u.username,
    u.first_name,
    u.last_name,
    u.role,
    u.last_login,
    COUNT(DISTINCT al.log_id) as total_actions,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN al.log_id END) as actions_last_7_days,
    COUNT(DISTINCT CASE WHEN al.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN al.log_id END) as actions_last_30_days
FROM auth_users u
LEFT JOIN auth_activity_log al ON u.user_id = al.user_id
GROUP BY u.user_id, u.username, u.first_name, u.last_name, u.role, u.last_login;

SELECT 'View created!' as Status;

-- ============================================================================
-- FINAL SUMMARY
-- ============================================================================

SELECT '========================================' as '';
SELECT '✓ INDEPENDENT TABLES CREATED!' as 'RESULT';
SELECT '========================================' as '';

SELECT 'New tables created with auth_ prefix:' as '';
SELECT '  - auth_users' as `Table_Name`;
SELECT '  - auth_sessions' as `Table_Name`;
SELECT '  - auth_activity_log' as `Table_Name`;
SELECT '  - auth_login_attempts' as `Table_Name`;

SELECT '' as '';
SELECT 'Your existing tables remain untouched:' as '';
SELECT '  - users (file upload system)' as `Table_Name`;
SELECT '  - user_sessions (file upload system)' as `Table_Name`;
SELECT '  - activity_log (file upload system)' as `Table_Name`;

SELECT '' as '';
SELECT 'IMPORTANT: You must now update PHP files to use new table names!' as 'CRITICAL';
SELECT 'See: UPDATE_PHP_FOR_RENAMED_TABLES.txt for instructions' as '';

SELECT '========================================' as '';
SELECT 'Default Admin Credentials:' as '';
SELECT 'Username: admin' as '';
SELECT 'Password: Admin@123' as '';
SELECT 'CHANGE IMMEDIATELY AFTER LOGIN!' as '';
SELECT '========================================' as '';
