-- Investor Database System Schema
-- MariaDB/MySQL Database

CREATE DATABASE IF NOT EXISTS investor_db;
USE investor_db;

-- Bank Staff Table
CREATE TABLE IF NOT EXISTS staff (
    staff_id INT AUTO_INCREMENT PRIMARY KEY,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    phone VARCHAR(20),
    department VARCHAR(100),
    position VARCHAR(100),
    active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (active),
    INDEX idx_name (last_name, first_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Clients Table
CREATE TABLE IF NOT EXISTS clients (
    client_id INT AUTO_INCREMENT PRIMARY KEY,

    -- Contact Information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    phone VARCHAR(20),
    mobile VARCHAR(20),
    address_line1 VARCHAR(200),
    address_line2 VARCHAR(200),
    city VARCHAR(100),
    state_province VARCHAR(100),
    postal_code VARCHAR(20),
    country VARCHAR(100),

    -- KYC Due Diligence
    kyc_status ENUM('Pending', 'In Progress', 'Completed', 'Expired') DEFAULT 'Pending',
    kyc_completion_date DATE,
    kyc_expiry_date DATE,
    id_type VARCHAR(50),
    id_number VARCHAR(100),
    id_expiry_date DATE,
    tax_id VARCHAR(50),
    pep_status ENUM('Yes', 'No', 'Unknown') DEFAULT 'Unknown',
    risk_rating ENUM('Low', 'Medium', 'High') DEFAULT 'Medium',

    -- Personal Information
    date_of_birth DATE,
    age INT,
    nationality VARCHAR(100),
    occupation VARCHAR(150),
    investor_profile TEXT,

    -- Financial Information
    annual_income DECIMAL(15,2),
    income_currency VARCHAR(10) DEFAULT 'USD',
    source_of_income TEXT,
    total_net_worth DECIMAL(15,2),
    net_worth_currency VARCHAR(10) DEFAULT 'USD',
    liquid_assets DECIMAL(15,2),

    -- Investment Information
    current_investments TEXT,
    investment_experience ENUM('None', 'Limited', 'Moderate', 'Extensive') DEFAULT 'None',
    investment_interests TEXT,
    investment_objectives TEXT,
    risk_tolerance ENUM('Conservative', 'Moderate', 'Aggressive') DEFAULT 'Moderate',
    investment_horizon ENUM('Short-term', 'Medium-term', 'Long-term') DEFAULT 'Medium-term',

    -- Additional Information
    notes TEXT,
    client_status ENUM('Prospect', 'Active', 'Inactive', 'Closed') DEFAULT 'Prospect',
    assigned_staff_id INT,

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_contact_date TIMESTAMP NULL,

    FOREIGN KEY (assigned_staff_id) REFERENCES staff(staff_id) ON DELETE SET NULL,
    INDEX idx_name (last_name, first_name),
    INDEX idx_email (email),
    INDEX idx_status (client_status),
    INDEX idx_kyc_status (kyc_status),
    INDEX idx_assigned_staff (assigned_staff_id),
    INDEX idx_investment_interests (investment_interests(100))
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Contact Records Table
CREATE TABLE IF NOT EXISTS contact_records (
    contact_id INT AUTO_INCREMENT PRIMARY KEY,
    client_id INT NOT NULL,
    staff_id INT NOT NULL,
    contact_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    contact_type ENUM('Newsletter', 'Phone Call', 'Visit', 'Email', 'Video Conference', 'Other') NOT NULL,
    subject VARCHAR(200),
    description TEXT,
    outcome TEXT,
    follow_up_required TINYINT(1) DEFAULT 0,
    follow_up_date DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (client_id) REFERENCES clients(client_id) ON DELETE CASCADE,
    FOREIGN KEY (staff_id) REFERENCES staff(staff_id) ON DELETE RESTRICT,
    INDEX idx_client (client_id),
    INDEX idx_staff (staff_id),
    INDEX idx_contact_date (contact_date),
    INDEX idx_contact_type (contact_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert some sample staff members
INSERT INTO staff (first_name, last_name, email, phone, department, position) VALUES
('John', 'Smith', 'john.smith@investbank.com', '+1-555-0101', 'Client Relations', 'Senior Advisor'),
('Sarah', 'Johnson', 'sarah.johnson@investbank.com', '+1-555-0102', 'Client Relations', 'Investment Advisor'),
('Michael', 'Brown', 'michael.brown@investbank.com', '+1-555-0103', 'Wealth Management', 'Portfolio Manager'),
('Emily', 'Davis', 'emily.davis@investbank.com', '+1-555-0104', 'Client Relations', 'Junior Advisor'),
('David', 'Wilson', 'david.wilson@investbank.com', '+1-555-0105', 'Compliance', 'KYC Officer');
